<?php
/**
 * Plugin Name: Cyan Stats
 * Plugin URI: https://cyanstats.com/docs/wordpress-plugin
 * Description: Integrates Cyan Stats web analytics into your website. Used by many solopreneurs and small businesses to track website performance and user behavior, Cyan Stats brings easy to read insights to help you grow your online presence.
 * Version: 0.2.0
 * Author: Rogerio Taques
 * Author URI: https://cyanstats.com
 * License: GPL2
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$VERSION = trim(file_get_contents(plugin_dir_path(__FILE__) . 'version.txt'));

/**
 * Plugin activation hook
 */
function cyanstats_activation_redirect( $plugin ) {
    if( $plugin == plugin_basename( __FILE__ ) ) {
        exit( wp_redirect( admin_url( 'admin.php?page=cyanstats-settings' ) ) );
    }
}
add_action( 'activated_plugin', 'cyanstats_activation_redirect' );

/**
 * Install the Cyan Stats tracking snippet in the header
 */
function cyanstats_install_snippet() {
    $project_token = get_option( 'cyanstats_project_token', '' );
    $data_token = $project_token ? ' data-token="' . esc_attr( $project_token ) . '"' : '';
    echo "<script async data-scroll=\"on\"{$data_token} id=\"cyanstats-script\" src=\"https://cyanstats.com/cs.js\"></script>";
}
add_action( 'wp_head', 'cyanstats_install_snippet' );

/**
 * Check for plugin updates
 */
function cyanstats_check_for_updates() {
    // Only check for updates once per day
    $last_check = get_option( 'cyanstats_last_update_check', 0 );
    if ( time() - $last_check < 86400 ) { // 24 hours
        return;
    }

    $remote_version_url = 'https://cyanstats.com/download/wordpress-plugin-version.txt';
    $response = wp_remote_get( $remote_version_url, array( 'timeout' => 10 ) );

    if ( is_wp_error( $response ) ) {
        return;
    }

    $remote_version = trim( wp_remote_retrieve_body( $response ) );
    $current_version = trim( file_get_contents( plugin_dir_path( __FILE__ ) . 'version.txt' ) );

    // Update last check time
    update_option( 'cyanstats_last_update_check', time() );

    if ( version_compare( $remote_version, $current_version, '>' ) ) {
        update_option( 'cyanstats_update_available', $remote_version );
    } else {
        delete_option( 'cyanstats_update_available' );
    }
}

/**
 * Display update notification in admin
 */
function cyanstats_admin_update_notice() {
    $update_version = get_option( 'cyanstats_update_available' );
    if ( ! $update_version || ! current_user_can( 'update_plugins' ) ) {
        return;
    }

    $current_version = trim( file_get_contents( plugin_dir_path( __FILE__ ) . 'version.txt' ) );
    ?>
    <div class="notice notice-info is-dismissible">
        <p>
            <strong>Cyan Stats Plugin Update Available!</strong>
            A new version (<?php echo esc_html( $update_version ); ?>) is available.
            You are currently using version <?php echo esc_html( $current_version ); ?>.
            <a href="https://cyanstats.com/download/wordpress-plugin" target="_blank" class="button button-secondary" style="margin-left: 10px;">Download Update</a>
            <button type="button" class="button-link" onclick="cyanstatsDismissNotice()" style="margin-left: 10px;">Dismiss</button>
        </p>
    </div>
    <script>
    function cyanstatsDismissNotice() {
        document.querySelector('.notice').style.display = 'none';
        fetch('<?php echo admin_url( 'admin-ajax.php' ); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: 'action=cyanstats_dismiss_update_notice&nonce=<?php echo wp_create_nonce( 'cyanstats_dismiss_update' ); ?>'
        });
    }
    </script>
    <?php
}

/**
 * Handle AJAX request to dismiss update notice
 */
function cyanstats_dismiss_update_notice() {
    if ( ! wp_verify_nonce( $_POST['nonce'], 'cyanstats_dismiss_update' ) || ! current_user_can( 'update_plugins' ) ) {
        wp_die( 'Unauthorized' );
    }

    delete_option( 'cyanstats_update_available' );
    wp_die();
}

/**
 * Check if current user has access to Cyan Stats
 */
function cyanstats_user_can_access() {
    $allowed_roles = get_option( 'cyanstats_allowed_roles', array( 'administrator' ) );

    if ( empty( $allowed_roles ) ) {
        return false;
    }

    $user = wp_get_current_user();
    return array_intersect( $allowed_roles, $user->roles );
}

/**
 * Add admin menu for Cyan Stats settings
 */
function cyanstats_add_admin_menu() {
    // Only show menu if user has access
    if ( ! cyanstats_user_can_access() ) {
        return;
    }

    add_menu_page(
        '',                           // Page title (Empty)
        'Cyan Stats',                 // Menu title
        'read',                       // Capability (minimal since we check roles above)
        'cyanstats-dashboard',        // Menu slug
        'cyanstats_dashboard_page',   // Callback function
        'dashicons-chart-line',       // Icon
        30                            // Position (after Users)
    );

    // Add submenu for settings
    add_submenu_page(
        'cyanstats-dashboard',        // Parent slug
        'Cyan Stats Settings',        // Page title
        'Settings',                   // Menu title
        'manage_options',             // Capability (only admins can change settings)
        'cyanstats-settings',         // Menu slug
        'cyanstats_settings_page'     // Callback function
    );
}
add_action( 'admin_menu', 'cyanstats_add_admin_menu' );

/**
 * Initialize settings
 */
function cyanstats_settings_init() {
    // Register settings
    register_setting( 'cyanstats_settings', 'cyanstats_project_token', array(
        'type' => 'string',
        'sanitize_callback' => 'cyanstats_sanitize_project_token',
        'default' => ''
    ) );

    register_setting( 'cyanstats_settings', 'cyanstats_embed_secret', array(
        'type' => 'string',
        'sanitize_callback' => 'cyanstats_sanitize_embed_secret',
        'default' => ''
    ) );

    register_setting( 'cyanstats_settings', 'cyanstats_allowed_roles', array(
        'type' => 'array',
        'sanitize_callback' => 'cyanstats_sanitize_allowed_roles',
        'default' => array( 'administrator' )
    ) );

    // Add settings section
    add_settings_section(
        'cyanstats_settings_section',
        __( '', 'cyanstats' ),
        'cyanstats_settings_section_callback',
        'cyanstats_settings'
    );

    // Add settings fields
    add_settings_field(
        'cyanstats_project_token',
        __( 'Project Token', 'cyanstats' ),
        'cyanstats_project_token_render',
        'cyanstats_settings',
        'cyanstats_settings_section'
    );

    add_settings_field(
        'cyanstats_embed_secret',
        __( 'Embed Secret', 'cyanstats' ),
        'cyanstats_embed_secret_render',
        'cyanstats_settings',
        'cyanstats_settings_section'
    );

    add_settings_field(
        'cyanstats_allowed_roles',
        __( 'Allowed User Roles', 'cyanstats' ),
        'cyanstats_allowed_roles_render',
        'cyanstats_settings',
        'cyanstats_settings_section'
    );
}
add_action( 'admin_init', 'cyanstats_settings_init' );

/**
 * Enqueue admin assets for Cyan Stats pages
 */
function cyanstats_enqueue_admin_assets( $hook ) {
    // Only load on Cyan Stats admin pages
    if ( $hook !== 'toplevel_page_cyanstats-dashboard' ) {
        return;
    }

    $plugin_url = plugin_dir_url( __FILE__ );

    wp_enqueue_script(
        'cyanstats-script',
        $plugin_url . 'assets/index.js',
        array(),
        $VERSION,
        true
    );

    wp_enqueue_style(
        'cyanstats-style',
        $plugin_url . 'assets/index.css',
        array(),
        $VERSION
    );

    wp_enqueue_style(
        'cyanstats-base-style',
        $plugin_url . 'assets/base.css',
        array(),
        $VERSION
    );
}
add_action( 'admin_enqueue_scripts', 'cyanstats_enqueue_admin_assets' );

/**
 * Modify script tag to add type="module" for Vue
 */
function cyanstats_add_module_type( $tag, $handle, $src ) {
    if ( 'cyanstats-script' !== $handle ) {
        return $tag;
    }

    // Change the script tag by adding type="module" and return it.
    $tag = '<script type="module" src="' . esc_url( $src ) . '" crossorigin></script>';
    return $tag;
}
add_filter( 'script_loader_tag', 'cyanstats_add_module_type', 10, 3 );

/**
 * Sanitize the project token input
 */
function cyanstats_sanitize_project_token( $input ) {
    // Remove any non-alphanumeric characters and limit to 12 characters
    $sanitized = preg_replace( '/[^a-zA-Z0-9]/', '', $input );
    return substr( $sanitized, 0, 12 );
}

/**
 * Sanitize the embed secret input
 */
function cyanstats_sanitize_embed_secret( $input ) {
    // Remove any non-alphanumeric characters and limit to 64 characters
    $sanitized = preg_replace( '/[^a-zA-Z0-9_\.-]/', '', $input );
    return substr( $sanitized, 0, 64 );
}

/**
 * Sanitize the allowed roles input
 */
function cyanstats_sanitize_allowed_roles( $input ) {
    if ( ! is_array( $input ) ) {
        return array( 'administrator' );
    }

    // Get all available WordPress roles
    $all_roles = wp_roles()->get_names();
    $sanitized = array();

    foreach ( $input as $role ) {
        if ( array_key_exists( $role, $all_roles ) ) {
            $sanitized[] = $role;
        }
    }

    // Always ensure administrator has access
    if ( ! in_array( 'administrator', $sanitized ) ) {
        $sanitized[] = 'administrator';
    }

    return $sanitized;
}

/**
 * Settings section callback
 */
function cyanstats_settings_section_callback() {
    echo '<p>' . __( 'Configure your Cyan Stats project for this website. By default, the website is automatically tracked by Cyan Stats when this plugin is active. But, in order to send traffic from a subdomain, you must add the data-token attribute to the script tag. The token can be found in the Settings of your Cyan Stats project dashboard.', 'cyanstats' ) . '</p>';
    echo '<p>' . __( 'The Embed Secret is required to access the dashboard data through the API. You can generate this secret in your Cyan Stats project settings.', 'cyanstats' ) . '</p>';
    echo '<p>' . __( 'Use the Allowed User Roles setting to control which user types can access the Cyan Stats dashboard.', 'cyanstats' ) . '</p>';
}

/**
 * Render the project token field
 */
function cyanstats_project_token_render() {
    $project_token = get_option( 'cyanstats_project_token', '' );
    ?>
    <input type="text"
        name="cyanstats_project_token"
        value="<?php echo esc_attr( $project_token ); ?>"
        maxlength="12"
        pattern="[a-zA-Z0-9]{1,12}"
        class="regular-text"
        placeholder="<?php esc_attr_e( 'Enter your project token', 'cyanstats' ); ?>"
        required
    />
    <p class="description">
        <?php _e( 'Up to 12 alphanumeric characters. Used for subdomain tracking.', 'cyanstats' ); ?>
    </p>
    <?php
}

/**
 * Render the embed secret field
 */
function cyanstats_embed_secret_render() {
    $embed_secret = get_option( 'cyanstats_embed_secret', '' );
    ?>
    <input type="text"
        name="cyanstats_embed_secret"
        value="<?php echo esc_attr( $embed_secret ); ?>"
        maxlength="64"
        pattern="[a-zA-Z0-9_.-]{1,64}"
        class="regular-text"
        placeholder="<?php esc_attr_e( 'Enter your embed secret', 'cyanstats' ); ?>"
        required
    />
    <p class="description">
        <?php _e( 'Up to 64 characters (letters, numbers, underscore, dash, dot). Required for dashboard access.', 'cyanstats' ); ?>
    </p>
    <?php
}

/**
 * Render the allowed roles field
 */
function cyanstats_allowed_roles_render() {
    $allowed_roles = get_option( 'cyanstats_allowed_roles', array( 'administrator' ) );
    $all_roles = wp_roles()->get_names();
    ?>
    <fieldset>
        <?php foreach ( $all_roles as $role_key => $role_name ): ?>
            <label>
                <input type="checkbox"
                       name="cyanstats_allowed_roles[]"
                       value="<?php echo esc_attr( $role_key ); ?>"
                       <?php checked( in_array( $role_key, $allowed_roles ) ); ?>
                       <?php echo $role_key === 'administrator' ? 'onclick="return false;" checked="checked"' : ''; ?>
                />
                <?php echo esc_html( $role_name ); ?>
                <?php if ( $role_key === 'administrator' ): ?>
                    <em>(<?php _e( 'always required', 'cyanstats' ); ?>)</em>
                <?php endif; ?>
            </label><br>
        <?php endforeach; ?>
    </fieldset>
    <p class="description">
        <?php _e( 'Select which user roles can access the Cyan Stats dashboard. Administrators always have access and can modify these settings.', 'cyanstats' ); ?>
    </p>
    <?php
}

/**
 * Check if any array values are contained in a string
 */
function array_includes( $array, $string ) {
    foreach ( $array as $item ) {
        if ( strpos( $string, $item ) !== false ) return true;
    }

    return false;
}

/**
 * Render the dashboard page
 */
function cyanstats_dashboard_page() {
    // Check if user has access
    if ( ! cyanstats_user_can_access() ) {
        wp_die( __( 'You do not have sufficient permissions to access this page.', 'cyanstats' ) );
    }

    $dev_domains = ['localhost', '127.0.0.1', '.local', '.dev', '.test'];
    $project_token = get_option( 'cyanstats_project_token', '' );
    $project_embed_secret = get_option( 'cyanstats_embed_secret', '' );
    $site_url = get_site_url();
    $is_dev = array_includes( $dev_domains, $site_url );
    $project_url = $is_dev ? 'example.com' : preg_replace( '/^https?:\/\//', '', $site_url );
    $api_url = $is_dev ? 'http://localhost:3000/api' : 'https://cyanstats.com/api';
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

        <div
            id="cyanstats-dashboard"
            style="margin-top: 20px; min-height: calc(100vh - 200px); border: 1px solid #ddd; border-radius: 4px; padding: 20px;"
            data-project-domain="<?php echo esc_attr( $project_url ); ?>"
            data-project-domain-token="<?php echo esc_attr( $project_token ); ?>"
            data-project-embed-secret="<?php echo esc_attr( $project_embed_secret ); ?>"
            data-api-url="<?php echo esc_attr( $api_url ); ?>"
        >
            <div class="loading-spinner" style="display: flex; justify-content: center; align-items: center; height: 200px;">
                <p><?php _e( 'Loading dashboard...', 'cyanstats' ); ?></p>
            </div>
        </div>

        <?php // Assets are loaded via the cyanstats_enqueue_admin_assets function ?>

        <div class="cyanstats-actions" style="margin-top: 20px;">
            <a href="<?php echo admin_url( 'admin.php?page=cyanstats-settings' ); ?>" class="button button-secondary">
                <?php _e( 'Settings', 'cyanstats' ); ?>
            </a>
            <a href="https://cyanstats.com?ref=wordpress-plugin" target="_blank" class="button button-secondary">
                <?php _e( 'Visit Cyan Stats', 'cyanstats' ); ?>
            </a>
            <a href="https://cyanstats.com/docs/wordpress-plugin?ref=wordpress-plugin" target="_blank" class="button button-secondary">
                <?php _e( 'Plugin Documentation', 'cyanstats' ); ?>
            </a>
        </div>
    </div>
    <?php
}

/**
 * Render the settings page
 */
function cyanstats_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

        <?php
        // Show success message if settings were updated
        if ( isset( $_GET['settings-updated'] ) ) {
            add_settings_error( 'cyanstats_messages', 'cyanstats_message', __( 'Settings saved successfully!', 'cyanstats' ), 'updated' );
        }

        // Display any error messages
        settings_errors( 'cyanstats_messages' );
        ?>

        <form action="options.php" method="post">
            <?php
            settings_fields( 'cyanstats_settings' );
            do_settings_sections( 'cyanstats_settings' );
            submit_button();
            ?>
        </form>

        <div class="cyanstats-info" style="margin-top: 20px; padding: 15px; background: #f9f9f9; border-left: 4px solid #00a0d2;">
            <h3><?php _e( 'About Cyan Stats', 'cyanstats' ); ?></h3>
            <p><?php _e( 'Cyan Stats provides easy-to-read analytics to help you understand your website performance and user behavior.', 'cyanstats' ); ?></p>
            <p>
                <a href="https://cyanstats.com?ref=wordpress-plugin" target="_blank" class="button button-secondary">
                    <?php _e( 'Visit Cyan Stats', 'cyanstats' ); ?>
                </a>
                <a href="https://cyanstats.com/docs/wordpress-plugin?ref=wordpress-plugin" target="_blank" class="button button-secondary">
                    <?php _e( 'Plugin Documentation', 'cyanstats' ); ?>
                </a>
            </p>
        </div>
    </div>
    <?php
}

/**
 * Add settings link to plugin actions
 */
function cyanstats_add_settings_link( $links ) {
    $settings_link = '<a href="' . admin_url( 'admin.php?page=cyanstats-settings' ) . '">' . __( 'Settings', 'cyanstats' ) . '</a>';
    array_unshift( $links, $settings_link );
    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'cyanstats_add_settings_link' );

// Add action hooks for update checking
add_action( 'admin_init', 'cyanstats_check_for_updates' );
add_action( 'admin_notices', 'cyanstats_admin_update_notice' );
add_action( 'wp_ajax_cyanstats_dismiss_update_notice', 'cyanstats_dismiss_update_notice' );
